/************************************************************************************
* Include
************************************************************************************/
#include "gap_interface.h"
/* Connection Manager */
#include "ble_conn_manager.h"

#include "ntag_bridge.h"
#include "ntag_driver_intern.h"
#include "ntag_defines.h"
#include "ntag_driver.h"
#include "nfc_device.h"

#include "Flash_Adapter.h"
#include "nvds.h"

#include "FunctionLib.h"
#include "app_ntag.h"
/************************************************************************************
* Private macros
************************************************************************************/
#define USE_NDEF_LOCAL_NAME 1
/************************************************************************************
* Private type definitions
************************************************************************************/
#define NDEF_DATA_BUFF_LEN 256
#define NO_ERROR 0x00U
#define ERROR_STATE NO_ERROR + 1

#if (USE_NDEF_LOCAL_NAME == 1)
#define ERROR_STATE_NO_NAME NO_ERROR + 2
#endif

#define AAR_FLAGS_INIT_FLAGS 0x00

typedef enum
{
    NDEF_RW_NO_ERROR = 0x00,
    NDEF_RW_ERR_NOT_T2T_FORMAT,
    NDEF_RW_ERR_TERM_TLV_MISSING,
    NDEF_RW_ERR_TOO_LONG_MSG,
    NDEF_RW_ERR_READ,
    NDEF_RW_ERR_WRITE
} NDEF_RW_STATUS_T;

/************************************************************************************
* Private memory declarations
************************************************************************************/
#ifndef USE_NTAG_I2C
NTAG_HANDLE_T ntag_handle; // NTAG handle
#endif

#if (USE_NDEF_LOCAL_NAME == 1)
static uint8_t AdvNameData[32];
static uint8_t LocalNameLength;
static gapAdStructure_t NDEF_advScanStruct[3];
const uint8_t  BleCarrierConfigurationRecord[32] = 
            { 'a','p','p','l','i','c','a','t','i','o','n','/','v','n','d','.','b','l','u','e','t','o','o','t','h','.','l','e','.','o','o','b'};
#endif
/************************************************************************************
* Private functions prototypes
************************************************************************************/

#if (USE_NDEF_LOCAL_NAME == 1)
/*******************************************************************************
* FUNCTION: NFC_MsgWrite
*
* Description: read NTAG 
*
*******************************************************************************/
NDEF_RW_STATUS_T NFC_MsgRead(uint8_t * pOutBuff, uint32_t * drOutBuffLen)
{
	uint16_t rOutBuffLenTmp;
    uint8_t bTemp;
	uint8_t aData_Block_1[NTAG_I2C_BLOCK_SIZE];
	
    /* Read the first block of the NDEF message. This consists of the NDEF header */
    if (NFC_ReadBlock(ntag_handle, NTAG_MEM_BLOCK_START_USER_MEMORY, aData_Block_1, NTAG_I2C_BLOCK_SIZE))
    {
        return NDEF_RW_ERR_READ;
    }
    
    /* fill the NDEF message control char */
    if(aData_Block_1[0] == NDEF_MESSAGE_TLV)
    {
        if(aData_Block_1[1] == NDEF_MSG_LEN_1B_TLV)
        {   /* get TLV length */
            rOutBuffLenTmp  = (((uint16_t)aData_Block_1[2] << 8) & 0x00FF);
            rOutBuffLenTmp &= ((uint16_t)aData_Block_1[3] & 0x00FF);
            *drOutBuffLen = (uint32_t)rOutBuffLenTmp;          
            /* copy the data to the pOutBuff from block 1 */
            FLib_MemCpy(pOutBuff, &aData_Block_1[4], (NTAG_I2C_BLOCK_SIZE-4));
            /* recalculate the length of the NDEF message need read*/
            rOutBuffLenTmp -= (NTAG_I2C_BLOCK_SIZE-4);				
            pOutBuff += (NTAG_I2C_BLOCK_SIZE-4);
        }
        else
        {   /* get TLV length */
            rOutBuffLenTmp = (uint16_t)aData_Block_1[1];
            *drOutBuffLen = (uint32_t)rOutBuffLenTmp;
            /* copy the data to the pOutBuff from block 1 */            
            if(rOutBuffLenTmp < (NTAG_I2C_BLOCK_SIZE-2))
            {
                FLib_MemCpy(pOutBuff, &aData_Block_1[2], rOutBuffLenTmp);
                if(aData_Block_1[2 + rOutBuffLenTmp] != TERMINATOR_TLV)
                {  return NDEF_RW_ERR_TERM_TLV_MISSING;  }
                rOutBuffLenTmp = 0;
            }
            else
            {
                FLib_MemCpy(pOutBuff, &aData_Block_1[2], (NTAG_I2C_BLOCK_SIZE-2));
                /* recalculate the length of the NDEF message need read*/
                rOutBuffLenTmp -= (NTAG_I2C_BLOCK_SIZE-2);
                pOutBuff += (NTAG_I2C_BLOCK_SIZE-2);                
            }
        }
    }
    else
    {
        return NDEF_RW_ERR_TERM_TLV_MISSING;
    }
    
    bTemp = rOutBuffLenTmp / NTAG_I2C_BLOCK_SIZE;
    if(bTemp < 56)
    {
        /* check if length is equal the full user memory range */
        if(bTemp == 55)
        {
            if ((rOutBuffLenTmp % NTAG_I2C_BLOCK_SIZE))
            {
                /* length is 55,xx blocks */
                return NDEF_RW_ERR_TOO_LONG_MSG;
            }
        }
    }
    else
    {
        return NDEF_RW_ERR_TOO_LONG_MSG;
    }

    /* Read NDEF message*/
    if(NFC_ReadBytes(ntag_handle, NTAG_I2C_BLOCK_SIZE*2, pOutBuff, rOutBuffLenTmp))
    {
        return NDEF_RW_ERR_READ;
    }
    /* Check TERMINATOR_TLV flag*/
    if(NFC_ReadBytes(ntag_handle, NTAG_I2C_BLOCK_SIZE*2 + rOutBuffLenTmp, &bTemp, 1))
    {
        return NDEF_RW_ERR_READ;
    }
    if(bTemp != TERMINATOR_TLV)
    {
        return NDEF_RW_ERR_TERM_TLV_MISSING;
    }

    return NDEF_RW_NO_ERROR;
}



#endif
/*******************************************************************************
* FUNCTION: NFC_MsgWrite
*
* Description:
*
*******************************************************************************/
NDEF_RW_STATUS_T NFC_MsgWrite(uint8_t * pOutBuff, uint32_t dwOutBuffLen)
{
    bool_t boRemainData = TRUE;
    uint8_t aData_Block_1[NTAG_I2C_BLOCK_SIZE];
    uint16_t wOutBuffLenTmp;
    uint8_t bTemp;
    /* check the length of the NDEF message, which will be written */
    if((dwOutBuffLen + 3) > NDEF_MSG_LEN_MAX_2K)
    {
        return NDEF_RW_ERR_TOO_LONG_MSG;
    }
    ntag_handle->status = NTAG_OK;
    /* set the zero data to the buffer for block 1 */
    FLib_MemSet(aData_Block_1, 0, NTAG_I2C_BLOCK_SIZE);
    /* fill the NDEF message control char */
    aData_Block_1[0] = NDEF_MESSAGE_TLV;
    /* calculate the TLV length */
    if((dwOutBuffLen +3) >= NDEF_MSG_LEN_1B_TLV)
    {
        /* retype the length of the message*/
        wOutBuffLenTmp = (uint16_t)dwOutBuffLen;
        /* fill the TLV message length */
        aData_Block_1[1] = (uint8_t)NDEF_MSG_LEN_1B_TLV;
        aData_Block_1[2] = (uint8_t)((wOutBuffLenTmp & 0xFF00) >> 8);
        aData_Block_1[3] = (uint8_t)(wOutBuffLenTmp & 0x00FF);
        /* copy the remain data to the buffer for block 1 */
        FLib_MemCpy(&aData_Block_1[4], pOutBuff, (NTAG_I2C_BLOCK_SIZE-4) );
        /* recalculate the remain length of the NDEF message */
        wOutBuffLenTmp -= (NTAG_I2C_BLOCK_SIZE-4);				
        pOutBuff += (NTAG_I2C_BLOCK_SIZE-4);		
    }		
    else		
    {
        /* fill the TLV message length */
        aData_Block_1[1] = (uint8_t)dwOutBuffLen;
        /* copy the remain data to the buffer for block 1 */
        if(aData_Block_1[1]>(NTAG_I2C_BLOCK_SIZE-2))
        {
            FLib_MemCpy(&aData_Block_1[2], pOutBuff, (NTAG_I2C_BLOCK_SIZE-2) );
            /* recalculate the remain length of the NDEF message */
            wOutBuffLenTmp = aData_Block_1[1] - (NTAG_I2C_BLOCK_SIZE-2);
            pOutBuff += (NTAG_I2C_BLOCK_SIZE-2); // 17f74
        }
        else
        {
            FLib_MemCpy(&aData_Block_1[2], pOutBuff, aData_Block_1[1] );
            aData_Block_1[aData_Block_1[1]+2] = TERMINATOR_TLV;
            boRemainData = FALSE;
            /* remain length of the NDEF message is 0 */
            wOutBuffLenTmp = 0;
            /* whole NDEF message + LTV bytes fits into the first user memory block */
            if (NFC_WriteBlock(ntag_handle, 1, aData_Block_1, NTAG_I2C_BLOCK_SIZE))
            {
                return NDEF_RW_ERR_WRITE;
            }
            if (NFC_WriteBlock(ntag_handle, 2, Null_Block, NTAG_I2C_BLOCK_SIZE))
            {
                return NDEF_RW_ERR_WRITE;
            }
        }
    }
    if (boRemainData)
    {
        bTemp = wOutBuffLenTmp / NTAG_I2C_BLOCK_SIZE;
        /* check if longer message then available memory size */
        if (bTemp < 56)
        {
            /* check if length is equal the full user memory range */
            if (bTemp == 55)
            {
                if ((wOutBuffLenTmp % NTAG_I2C_BLOCK_SIZE))
                {
                    /* length is 55,xx blocks */
                    boRemainData = FALSE;
                }
            }
        }
        else
        {
            /* message is longer than 55x16 bytes (> 1k version) */
            boRemainData = FALSE;
        }
    }
    if (boRemainData)
    {
        /* fill the terminator char */
        *(pOutBuff+wOutBuffLenTmp)= TERMINATOR_TLV;
        /* Write the default "Empty NDEF" message to the NTAG chip
         * This is from reason some error during the NFC data transmission. The header of th
         * with NDEF message data will be written at the end of the data transmission.
         * */
        if (NFC_WriteBytes( ntag_handle, NTAG_MEM_ADRR_I2C_ADDRESS, Default_Empty_NDEF_msg, NTAG_I2C_BLOCK_SIZE*2))
	      {
            return NDEF_RW_ERR_WRITE;
        }
	      /* Write remain blocks of the NDEF message. */
        if (NFC_WriteBytes(ntag_handle, NTAG_I2C_BLOCK_SIZE*2, pOutBuff, wOutBuffLenTmp+1))
        {
            return NDEF_RW_ERR_WRITE;
        }
        /* Write the first block of the NDEF message. This consists of the NDEF header */
        if (NFC_WriteBlock(ntag_handle, 1, aData_Block_1, NTAG_I2C_BLOCK_SIZE))
        {
            return NDEF_RW_ERR_WRITE;
        }
    }
    
    return NDEF_RW_NO_ERROR;
}



#if (USE_NDEF_LOCAL_NAME == 1)
/*******************************************************************************
* FUNCTION: NFC_MsgWrite
*
* Description:
*
*******************************************************************************/
status_t NDEF_Get_Ble_Adv_Local_Name(uint8_t * AdvName, uint8_t * len)
{
    uint8_t NDEF_Read_msg[NDEF_MSG_LEN_MAX_1K];
    uint32_t NDEFMsgLen;
    status_t Status = NO_ERROR;
    NDEF_RW_STATUS_T NfcStatus;

    /*Read NDEF message*/
    NfcStatus = NFC_MsgRead(NDEF_Read_msg, &NDEFMsgLen);
    if (NfcStatus != NDEF_RW_NO_ERROR) 
    { 
        return ERROR_STATE;
    }

    /*check Bluetooth Carrier Configuration Record*/
    if(FLib_MemCmp(BleCarrierConfigurationRecord, &NDEF_Read_msg[3], NDEF_Read_msg[1]))
    {
        uint32_t i;
        /*get TLV message start point*/
        i = (uint32_t)NDEF_Read_msg[1] + 3; 
        
        while(i < NDEFMsgLen)
        {   /*Check whether the current TLV store the BLE local name*/
            if((NDEF_Read_msg[i+1] != NDEF_BLE_LOCAL_NAME0) && (NDEF_Read_msg[i+1] != NDEF_BLE_LOCAL_NAME1))
            {   /*current TLV not store the BLE local name*/
                /*Check next TLV*/
                i += (NDEF_Read_msg[i] + 1);
            }
            else
            {
                /*current TLV are storing the BLE local name*/
                /*copy BLE local name to static buffer*/
                FLib_MemCpy(AdvName,&NDEF_Read_msg[i + 2],NDEF_Read_msg[i] - 1); 
                *len = NDEF_Read_msg[i] - 1;         
                break;
            }
        }
        
        if( i < NDEFMsgLen)
        {
            Status = NO_ERROR;
        }
        else
        {
            /*There are no BLE local name store in NDEF*/
            Status = ERROR_STATE_NO_NAME;
        }
    }
    else
    {
        Status = ERROR_STATE;
    }  

    return  Status;   
}
#endif


/************************************************************************************
* Public functions
************************************************************************************/

/*******************************************************************************
* FUNCTION: NDEF_Pairing_Write
*
* Description:
*
*******************************************************************************/
status_t NDEF_Pairing_Write(void)
{
    uint8_t n;
    uint16_t bd_address_len = NVDS_LEN_BD_ADDRESS;
    status_t Status = NO_ERROR;
    NDEF_RW_STATUS_T NfcStatus;
    hardwareParameters_t pHwParams;
    /*default NDEF message*/
    uint8_t BLE_pairing_NDEF_msg1[100] = 
            { 
            /* 3 */ 0xD2, 0x20,
#if (USE_NDEF_LOCAL_NAME == 1)
                    0x15,
#else
                    0x0c,  
#endif                        
            /* 32 */ 'a','p','p','l','i','c','a','t','i','o','n','/','v','n','d','.','b','l','u','e','t','o','o','t','h','.','l','e','.','o','o','b', 
            /* 1 */ 0x08, // LE Bluetooth Device Address length: 8 bytes 
            /* 1 */ 0x1B, // LE Bluetooth Device Address data type 
            /* 6 */ 0xc6, 0xc5, 0xc4, 0xc3, 0xc2, 0xc1, // Bluetooth Device Address: 6 bytes + 1 (next byte) 
            /* 1 */ 0x00, // added at the end of the Device Address 
            /* 1 */ 0x02, // LE Role Length: 2 bytes 
            /* 1 */ 0x1c, // LE Role data type 
            /* 1 */ 0x02, // LE Role: PeripheralPrefered 
#if (USE_NDEF_LOCAL_NAME == 1)
            /* 1 */ 0x08, // LE Local Name length 
            /* 1 */ 0x09, // LE Local Name data type 
            /* 7 */ 'F','S','L','_','H','I','D'                
#endif
            };
    
    /*read NVDS BD address*/			
    Status = nvds_get(NVDS_TAG_BD_ADDRESS, (uint16_t *)&bd_address_len, (uint8_t *)pHwParams.bluetooth_address);
    if(Status != NVDS_OK)
    {
        uint8_t vendor_bdaddress[6];
        FLib_MemCpy(vendor_bdaddress, (uint8_t *)FSL_FEATURE_FLASH_ADDR_OF_VENDOR_BD_ADDR, 6);
        FLib_MemCpyReverseOrder((uint8_t *)pHwParams.bluetooth_address, vendor_bdaddress, 6);
        Status = NO_ERROR;
    }
    
    /* copy the BD address to the NDEF message field */ 
    for (n = 0; n < 6; n++) 
    {
        BLE_pairing_NDEF_msg1[37+n] = pHwParams.bluetooth_address[n]; 
    }
    

#if (USE_NDEF_LOCAL_NAME == 1)
    gapAdStructure_t * p;
    
    p = (gapAdStructure_t *)gAppAdvertisingData.aAdStructures;
    for (n = 0; n < gAppAdvertisingData.cNumAdStructures ; n++)
    {
        FLib_MemCpy(NDEF_advScanStruct + n , gAppAdvertisingData.aAdStructures + n, sizeof(gapAdStructure_t)); 
    }    

    /*Find store BLE local name point*/                
    p = (gapAdStructure_t *)NDEF_advScanStruct;     
    for(n = 0; n < gAppAdvertisingData.cNumAdStructures; n++)
    {   
        if( p->adType == gAdShortenedLocalName_c)
        {
            break;
        }
        p++;
    }    
    
    /*get BLE adversting local name from NDEF*/
    Status = NDEF_Get_Ble_Adv_Local_Name(AdvNameData, &LocalNameLength);
    if(Status == NO_ERROR)
    {
        p->aData = AdvNameData;
        p->length = LocalNameLength + 1;
        gAppAdvertisingData.aAdStructures = (void *)NDEF_advScanStruct;
        /* Setup Advertising and scanning data */
        Gap_SetAdvertisingData(&gAppAdvertisingData, NULL);
        /*modify payload length*/
        BLE_pairing_NDEF_msg1[2] = 0x0e + LocalNameLength;
        /*modify LE Local Name length = local name length + local name data type(1byte)*/
        BLE_pairing_NDEF_msg1[47] = LocalNameLength + 1;
        /*copy the BLE local name to the write NDEF message field */
        FLib_MemCpy(&BLE_pairing_NDEF_msg1[49],AdvNameData,LocalNameLength);
    }

#endif  
    
    NfcStatus = NFC_MsgWrite(BLE_pairing_NDEF_msg1, BLE_pairing_NDEF_msg1[1] + BLE_pairing_NDEF_msg1[2] + 3);    
    if (NfcStatus != NDEF_RW_NO_ERROR) 
    { 
        Status = ERROR_STATE;
    }

    return Status;
}


/*******************************************************************************
* FUNCTION: NDEF_Demo_Write
*
* Description:
*
*******************************************************************************/
status_t NDEF_Demo_Write(void)
{
    status_t Status = NO_ERROR;
    NDEF_RW_STATUS_T NfcStatus;

    /*Default NDEF message: SmartPoster 
     * Title: NTAG I2C Explorer
     * Link: http://www.nxp.com/demoboard/OM5569 
     */
    uint8_t Default_SmartPoster[] = { 
                    0x91, 0x02, 0x35, 0x53, 0x70, 0x91, 0x01, 0x14, 0x54, 0x02, 0x65, 0x6E, 0x4E, 0x54, 
                    0x41, 0x47, 0x20, 0x49, 0x32, 0x43, 0x20, 0x45, 0x58, 0x50, 0x4C, 0x4F, 0x52, 0x45, 0x52, 0x51,
                    0x01, 0x19, 0x55, 0x01, 0x6E, 0x78, 0x70, 0x2E, 0x63, 0x6F, 0x6D, 0x2F, 0x64, 0x65, 0x6D, 0x6F,
                    0x62, 0x6F, 0x61, 0x72, 0x64, 0x2F, 0x4F, 0x4D, 0x35, 0x35, 0x36, 0x39, 0x54, 0x0F, 0x13, 0x61,
                    0x6E, 0x64, 0x72, 0x6F, 0x69, 0x64, 0x2E, 0x63, 0x6F, 0x6D, 0x3A, 0x70, 0x6B, 0x67, 0x63, 0x6F,
                    0x6D, 0x2E, 0x6E, 0x78, 0x70, 0x2E, 0x6E, 0x74, 0x61, 0x67, 0x69, 0x32, 0x63, 0x64, 0x65, 0x6D,
                    0x6F 
    };
		
    NfcStatus = NFC_MsgWrite(Default_SmartPoster, sizeof(Default_SmartPoster));
    if (NfcStatus != NDEF_RW_NO_ERROR) 
    { 
        Status = ERROR_STATE;
    }

    return Status;
}

/* *********************************************************************************
* @} end of file
********************************************************************************** */
